<?php


namespace App\Services;

use Illuminate\Support\Facades\DB;
use App\DataSheet;
use App\QuestionAnswer;
use App\ReportData;
use App\SheetAnswer;
use App\SheetQuestion;
use App\TaskReport;
use App\User;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use function Psy\debug;


class DataSheetService
{
    // data sheet listing page
    public function dataSheetPage(){
        $user = Auth::user() ;
        $branch_id='';
        if($user->role=='Admin'){
            $branch_id=$user->branch_id;
        }else if($user->role=='Executive'){
            $branch_id=$user->id;
        }
        $records = DataSheet::select('*');
        if(!empty($branch_id)){
            $records = $records->where("branch_id",$branch_id);
        }
        $records = $records->withCount("tasks")->get();

        return view("portal.dataSheets.index", compact("records"));
    }

    // add new data sheet request
    public function addNewDataSheetRequest($request){
        //if(DataSheet::count() >= 10){
            //return HelperServices::returnJsonResponse(2, 'Validation Errors', [], ['Maximum 10 sheets can be added.'], 422);
        //}
        $customMsgs = [
            'name.required' => 'Please enter sheet name.',
        ];
        $validator = Validator::make($request->all(),[
            'name'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = new DataSheet();
        $record->branch_id = Auth::user()->id;
        $record->name = $request->name;
        $record->add_additional_msg = $request->add_additional_msg;
        if($record->save()){
            $redirectUrl = route("portal.editDataSheetPage", ["id" => $record->id]);
            return HelperServices::returnJsonResponse(1, 'Data sheet created successfully.', ["redirect_url" => $redirectUrl]);
        } else {
            return HelperServices::returnJsonResponse(0, 'Data sheet not created.');
        }
    }

    // get data sheet page
    public function editDataSheet($id){
        $record = DataSheet::find($id);
        if(!$record){
            abort(404);
        }
        return view('portal.dataSheets.editDataSheet', compact('record'));
    }

    // update data sheet
    public function updateDataSheetRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'name.required' => 'Please enter sheet name.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'name'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = DataSheet::find($request->id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Record not found");
        }
        $record->name = $request->name;
        $record->add_additional_msg = $request->add_additional_msg;
        $record->status = $request->status;
        if($record->save()){
            return HelperServices::returnJsonResponse(1, "Record updated successfully");
        } else {
            return HelperServices::returnJsonResponse(0, "Nothing updated");
        }
    }

    // view data sheet page
    public function viewDataSheetPage($id){
        $record = DataSheet::find($id);
        if(!$record){
            abort(404);
        }
        $record = $record->load("tasks.answers");
        return view("portal.dataSheets.viewDataSheet", compact("record"));
    }

    // delete data sheet
    public function deleteDataSheet($id){
        $record = DataSheet::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // add sheet task page
    public function addSheetTaskPage($sheetId){
        $record = DataSheet::find($sheetId);
        if(!$record){
            abort(404);
        }
        $sheetId = $record->id;
        return view("portal.dataSheets.sheetTasks.addSheetTask", compact("sheetId"));
    }

    // add sheet task request
    public function addSheetTaskRequest($request){

        $customMsgs = [
            'data_sheet_id.required' => 'Error: Something went wrong.',
            'data_sheet_id.exists' => 'Error: Data Sheet not found.',
            'title.required' => 'Please enter title.',
            'progress_note.required' => 'Please enter progress note.',
            'progress_note_independent.required' => 'Please enter progress note.',
        ];
        $validator = Validator::make($request->all(),[
            'data_sheet_id'=>'required|exists:data_sheets,id',
            'title'=>'required',
            'progress_note'=>'required',
            'progress_note_independent'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }
        //if(SheetQuestion::where('data_sheet_id','=',$request->data_sheet_id)->count() >= 25){
            //return HelperServices::returnJsonResponse(2, 'Validation Errors', [], ['Maximum 25 tasks can be added.'], 422);
        //}

        $record = new SheetQuestion();
        $record->title = $request->title;
        $record->progress_note = $request->progress_note;
        $record->progress_note_independent = $request->progress_note_independent;
        $record->data_sheet_id = $request->data_sheet_id;
        if($record->save()){
            $redirectUrl = route("portal.editSheetTaskPage", ["id" => $record->id]);
            return HelperServices::returnJsonResponse(1, 'Record created successfully.', ["redirect_url" => $redirectUrl]);
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not created.');
        }
    }

    // edit sheet task page
    public function editSheetTaskPage($taskId){
        $record = SheetQuestion::find($taskId);
        if(!$record){
            abort(404);
        }
        return view("portal.dataSheets.sheetTasks.editSheetTask", compact("record"));
    }

    // update sheet task request
    public function updateSheetTaskRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'title.required' => 'Please enter title.',
            'progress_note.required' => 'Please enter progress note.',
            'progress_note_independent.required' => 'Please enter progress note.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'title'=>'required',
            'progress_note'=>'required',
            'progress_note_independent'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = SheetQuestion::find($request->id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        $record->title = $request->title;
        $record->progress_note = $request->progress_note;
        $record->progress_note_independent = $request->progress_note_independent;
        $record->status = $request->status;
        if($record->save()){
            return HelperServices::returnJsonResponse(1, 'Record updated successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not updated.');
        }
    }

    // delete sheet question
    public function deleteSheetTask($id){
        $record = SheetQuestion::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // add answer in task
    public function addAnswerInTaskRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'answer.required' => 'Please enter answer.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'answer'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $question = SheetQuestion::find($request->id);
        if(!$question){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }

        $record = new SheetAnswer();
        $record->answer = $request->answer;
        if($record->save()){
            $question->answers()->attach($record->id, ["delete_answer" => 1]);
            return HelperServices::returnJsonResponse(1, 'Record added successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not added.');
        }
    }

    // delete answer in task
    public function deleteAnswerInTask($id){
        $record = QuestionAnswer::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // get data sheet listing api
    public function activeDataSheetListingApi(){
        $branch_id = Auth::user()->branch_id;
        if(Auth::user()->role=="Executive"){
            $branch_id = Auth::user()->id;
        }

        $records = DataSheet::where("status", 1)->where("branch_id",$branch_id)->get();

        return HelperServices::returnJsonResponse(1, "Success", $records);
    }

    // get data sheet tasks api
    public function dataSheetTasks($id){
        $record = DataSheet::select("id", "name", "add_additional_msg")->where(["status" => 1, "id" => $id])->first();
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Data sheet not found.");
        }
        $record = $record->load(['tasks' => function ($query) {
                        $query->select('id', 'title', 'data_sheet_id')->where('status', 1);
                    }, "tasks.answers" => function ($query){
                        $query->select('sheet_answers.id', 'sheet_answers.answer');
                    }]);
        return HelperServices::returnJsonResponse(1, "Success", $record);
    }

    // generate progress sheet note api
    public function generateSheetProgressNote($request){
        $customMsgs = [
            'sheet_id.required' => 'Please select data sheet.',
            'client_id.required' => 'Please select client.',
            'sheet_data.required' => 'Error: task data not found.',
            'sheet_data.*.task_id.required' => 'Error: task id not selected.',
            'sheet_data.*.answer_id.required' => 'Please select answer.',
        ];

        $validator = Validator::make(json_decode(json_encode($request),1),[
            'sheet_id' => 'required',
            'client_id' => 'required',
            'sheet_data' => 'required',
            'sheet_data.*.task_id'=>'required',
            'sheet_data.*.answer_id'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 200);
        }

        // get data sheet record
        $record = DataSheet::where(["status" => 1, "id" => $request->sheet_id])->first();
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Data sheet not found.");
        }

        // get client data
        $client = User::where(["id" => $request->client_id, "status" => 1, "role" => "Client"])->first();
        if(!$client){
            return HelperServices::returnJsonResponse(0, "Client not found.");
        }
        $clientFullName = $client->name;
        list($clientFirstName) = explode(' ', $clientFullName);
        $clientGenderNoun = ($client->gender == "Male")? "He": "She";

        $clientHisHerGenderNoun = ($client->gender == "Male")? "His": "Her";

        // get question id from request
        $qtnArr = array_column($request->sheet_data, 'task_id');
        // get answer id from request
        $ansArr = array_column($request->sheet_data, 'answer_id');
        // get question data from db
        $questions_results = SheetQuestion::select("progress_note", "id","progress_note_independent")->whereIn("id", $qtnArr)->get()->toArray();

        /*Added by jehanzeb start*/
        $questions = [];
        foreach ($questions_results as $questions_row){
            $questions[$questions_row['id']] = $questions_row['progress_note'];
            $questions_ind[$questions_row['id']] = $questions_row['progress_note_independent'];
        }
        /*Added by jehanzeb end*/
        // get answer data from db
        $answers = SheetAnswer::whereIn("id", $ansArr)->pluck("answer", "id")->toArray();

        $progressNote = "<p>";
        $index = 0;

        foreach($request->sheet_data as $data){
            $data = (array)$data;
            $index++;

            $answerPrompt = (isset($answers[$data["answer_id"]]))? 'with '.strtolower($answers[$data["answer_id"]])." prompt. ": "";
            if($data["answer_id"] == '1'){
                $note = (isset($questions_ind[$data["task_id"]]))? $questions_ind[$data["task_id"]]: "";
            }else{
                $note = (isset($questions[$data["task_id"]]))? $questions[$data["task_id"]]: "";
            }

            $note = str_replace("##ClientFullName##", $clientFullName, $note);  // replace client full name
            $note = str_replace("##ClientFirstName##", $clientFirstName, $note);  // replace client first name
            $note = str_replace("##ClientGenderNoun##", $clientGenderNoun, $note);  // replace client gender noun
            $note = str_replace("##HIS-HER-GenderNoun##", $clientHisHerGenderNoun, $note);  // replace client HIS / HER gender noun
            $note = str_replace("##AnswerPrompt##", $answerPrompt, $note);  // replace selected answer prompt

            $progressNote .= $note." ";
            if($index % 3 == 0){
                $progressNote .= "</p><p>";
            }

        }
        /*$sender = Auth::user()->name;
        $executive =User::find(Auth::user()->branch_id);
        $data = array('name'=>$executive->name, "submittedBy"=>$sender,"progressNote"=>$progressNote,"email"=>'muhammadsajidjaved@gmail.com');

        Mail::send(['html'=>'mail'], $data, function($message) use($data) {
            $message->to($data['email'],$data['name'])->subject('Task Submitted By '.$data['submittedBy']);
            $message->from('info@bbcareadmin.com','BBCareAdmin');
        });*/

        $progressNote .= "</p>";
        $progressNote .= "<p>".$request->add_additional_msg."</p>";
        return HelperServices::returnJsonResponse(1, "Success", ["progress_note" => $progressNote]);
    }

    // save data sheet tasks & note api
    public function saveSheetTasks($request){
        $customMsgs = [
            'sheet_id.required' => 'Please select data sheet.',
            'client_id.required' => 'Please select client.',
            'sheet_data.required' => 'Error: task data not found.',
            'sheet_data.*.task_id.required' => 'Error: task id not selected.',
            'sheet_data.*.answer_id.required' => 'Please select answer.',
            'progress_note.required' => 'Error: progress note not found.',
        ];
        $request = json_decode(json_encode($request),1);
        $validator = Validator::make((array)$request,[
            'sheet_id' => 'required',
            'client_id' => 'required',
            'sheet_data' => 'required',
            'sheet_data.*.task_id'=>'required',
            'sheet_data.*.answer_id'=>'required',
            'progress_note' => 'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 200);
        }

        // get data sheet record
        $record = DataSheet::where(["status" => 1, "id" => $request['sheet_id']])->first();
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Data sheet not found.");
        }

        // get client data
        $client = User::where(["id" => $request['client_id'], "status" => 1, "role" => "Client"])->first();

        if(!$client){
            return HelperServices::returnJsonResponse(0, "Client not found.");
        }
        // save records in db
        $dbStatus = $this->saveTaskDataInDB($record->id, $client->id, $request['progress_note'], $request['sheet_data'], $request['add_additional_msg']);
        if($dbStatus){
            $sender = Auth::user()->name;
            $executive =User::find(Auth::user()->branch_id);
            $data = array('name'=>$executive->name, "submittedBy"=>$sender,"progressNote"=>$request['progress_note'],"email"=>$executive['email']);

            Mail::send(['html'=>'mail'], $data, function($message) use($data) {
                $message->to($data['email'],$data['name'])->subject('Task Submitted By '.$data['submittedBy']);
                $message->from('info@bbcareadmin.com','BBCareAdmin');
            });

            return HelperServices::returnJsonResponse(1, "Tasks added successfully");
        } else {
            return HelperServices::returnJsonResponse(0, "Something went wrong");
        }
    }

    // save task report data in database
    private function saveTaskDataInDB($sheetId, $clientId, $progressNote, $taskData = array(), $additionalMsg = ""){
        $currentUser = request()->user();
        $record = new TaskReport();
        $record->data_sheet_id = $sheetId;
        $record->staff_id = $currentUser->id;
        $record->client_id = $clientId;
        $record->additional_message = $additionalMsg;
        $record->progress_note = $progressNote;
        if($record->save()){

            $dataArr = [];
            foreach ($taskData as $data){
                $dataArr[] = new ReportData(["sheet_question_id" => $data["task_id"], "sheet_answer_id" => $data["answer_id"]]);
            }
            if(sizeof($dataArr) > 0) {
                $record->reportData()->saveMany($dataArr);
            }


            return true;
        } else {
            return false;
        }
    }
}
