<?php


namespace App\Services;

use Illuminate\Support\Facades\DB;
use App\DataSheet;
use App\QuestionAnswer;
use App\ReportData;
use App\SheetAnswer;
use App\SheetQuestion;
use App\TaskReport;
use App\User;
use App\SheetType;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Validator;
use function Psy\debug;
use App\Category;
use App\UserCategory;
use App\DataSheetCategory;


class DataSheetService
{
    // data sheet listing page
    public function dataSheetPage(){
        $user = Auth::user() ;
        $branch_id='';
        if($user->role=='Admin'){
            $branch_id=$user->branch_id;
        }else if($user->role=='Executive'){
            $branch_id=$user->id;
        }
        $records = DataSheet::select('*');
        if(!empty($branch_id)){
            $records = $records->where("branch_id",$branch_id);
        }
        $records = $records->withCount("tasks")->orderBy('id','ASC')->get();

        return view("portal.dataSheets.index", compact("records"));
    }

    // add data sheet page
    public function addDataSheet(){
        $categories = Category::where('status', 1)->get();
        return view("portal.dataSheets.addDataSheet", compact('categories'));
    }

    // add new data sheet request
    public function addNewDataSheetRequest($request){
        //if(DataSheet::count() >= 10){
            //return HelperServices::returnJsonResponse(2, 'Validation Errors', [], ['Maximum 10 sheets can be added.'], 422);
        //}
        $customMsgs = [
            'name.required' => 'Please enter sheet name.',
            'categories.required' => 'Please select at-least one category.',
        ];
        $validator = Validator::make($request->all(),[
            'name'=>'required',
            'categories'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = new DataSheet();
        $record->branch_id = Auth::user()->id;
        $record->name = $request->name;
        $record->add_additional_msg = $request->add_additional_msg;
        if($record->save()){
            $record->categories()->attach($request->categories);
            $redirectUrl = route("portal.editDataSheetPage", ["id" => $record->id]);
            return HelperServices::returnJsonResponse(1, 'Data sheet created successfully.', ["redirect_url" => $redirectUrl]);
        } else {
            return HelperServices::returnJsonResponse(0, 'Data sheet not created.');
        }
    }

    // get data sheet page
    public function editDataSheet($id){
        $record = DataSheet::find($id);
        if(!$record){
            abort(404);
        }
        return view('portal.dataSheets.editDataSheet', compact('record'));
    }

    // update data sheet
    public function updateDataSheetRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'name.required' => 'Please enter sheet name.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'name'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = DataSheet::find($request->id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Record not found");
        }
        $record->name = $request->name;
        $record->add_additional_msg = $request->add_additional_msg;
        $record->status = $request->status;
        if($record->save()){
            return HelperServices::returnJsonResponse(1, "Record updated successfully");
        } else {
            return HelperServices::returnJsonResponse(0, "Nothing updated");
        }
    }

    // view data sheet page
    public function viewDataSheetPage($id){
        $record = DataSheet::find($id);
        if(!$record){
            abort(404);
        }
        $record = $record->load("tasks.answers");
        $assignedCategories = DataSheetCategory::where('data_sheet_id', $id)->pluck("category_id");
        $categories = Category::where('status', 1)->whereNotIn('id', $assignedCategories)->get();
        return view("portal.dataSheets.viewDataSheet", compact("record", 'categories'));
    }

    // delete data sheet
    public function deleteDataSheet($id){
        $record = DataSheet::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // add sheet task page
    public function addSheetTaskPage($sheetId){
        $record = DataSheet::find($sheetId);
        if(!$record){
            abort(404);
        }
        $sheetId = $record->id;
        $sheetTypes = SheetType::selectRaw('sheet_types.id, sheet_types.title, GROUP_CONCAT(sheet_answers.answer) as options')
            ->join('sheet_answers', 'sheet_types.id', '=', 'sheet_answers.sheet_type_id')
            ->where(['sheet_types.status' => 1, 'sheet_types.deleted' => 0])
            ->groupBy('sheet_types.id')->get();
        return view("portal.dataSheets.sheetTasks.addSheetTask", compact("sheetId", 'sheetTypes'));
    }

    // add sheet task request
    public function addSheetTaskRequest($request){

        $customMsgs = [
            'data_sheet_id.required' => 'Error: Something went wrong.',
            'data_sheet_id.exists' => 'Error: Data Sheet not found.',
            'title.required' => 'Please enter title.',
            'progress_notes.required' => 'Please enter at-least one progress note.',
            'progress_notes_independent.required' => 'Please enter at-least one progress note.',
            'sheet_type.required' => 'Please select sheet type.',
        ];
        $validator = Validator::make($request->all(),[
            'data_sheet_id'=>'required|exists:data_sheets,id',
            'title'=>'required',
            'progress_notes'=>'required|array',
            'progress_notes.*' => 'required_with:progress_note.*',
            'progress_notes_independent'=>'required|array',
            'progress_notes_independent.*'=>'required_with:progress_notes_independent.*',
            'sheet_type' => 'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }
        //if(SheetQuestion::where('data_sheet_id','=',$request->data_sheet_id)->count() >= 25){
            //return HelperServices::returnJsonResponse(2, 'Validation Errors', [], ['Maximum 25 tasks can be added.'], 422);
        //}

        $record = new SheetQuestion();
        $record->title = $request->title;
        $record->progress_note = json_encode($request->progress_notes);
        $record->progress_note_independent = json_encode($request->progress_notes_independent);
        $record->data_sheet_id = $request->data_sheet_id;
        $record->sheet_type_id = $request->sheet_type;
        $record->assessment = $request->progress_assessment;
        $record->plan = $request->progress_plan;
        $record->independent_assessment = $request->independent_assessment;
        $record->independent_plan = $request->independent_plan;
        $record->is_new = 1;
        if($record->save()){
            $redirectUrl = route("portal.editSheetTaskPage", ["id" => $record->id]);
            return HelperServices::returnJsonResponse(1, 'Record created successfully.', ["redirect_url" => $redirectUrl]);
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not created.');
        }
    }

    // edit sheet task page
    public function editSheetTaskPage($taskId){
        $record = SheetQuestion::find($taskId);
        if(!$record){
            abort(404);
        }
        return view("portal.dataSheets.sheetTasks.editSheetTask", compact("record"));
    }

    // update sheet task request
    public function updateSheetTaskRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'title.required' => 'Please enter title.',
            'progress_notes.required' => 'Please enter at-least one progress note.',
            'progress_notes_independent.required' => 'Please enter at-least one progress note.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'title'=>'required',
            'progress_notes'=>'required|array',
            'progress_notes.*' => 'required_with:progress_note.*',
            'progress_notes_independent'=>'required|array',
            'progress_notes_independent.*'=>'required_with:progress_notes_independent.*',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = SheetQuestion::find($request->id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        $record->title = $request->title;
        $record->progress_note = json_encode($request->progress_notes);
        $record->progress_note_independent = json_encode($request->progress_notes_independent);
        $record->assessment = $request->progress_assessment;
        $record->plan = $request->progress_plan;
        $record->independent_assessment = $request->independent_assessment;
        $record->independent_plan = $request->independent_plan;
        $record->is_new = 1;
        $record->status = $request->status;
        if($record->save()){
            return HelperServices::returnJsonResponse(1, 'Record updated successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not updated.');
        }
    }

    // delete sheet question
    public function deleteSheetTask($id){
        $record = SheetQuestion::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // add answer in task
    public function addAnswerInTaskRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'answer.required' => 'Please enter answer.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'answer'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $question = SheetQuestion::find($request->id);
        if(!$question){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }

        $record = new SheetAnswer();
        $record->answer = $request->answer;
        if($record->save()){
            $question->answers()->attach($record->id, ["delete_answer" => 1]);
            return HelperServices::returnJsonResponse(1, 'Record added successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not added.');
        }
    }

    // delete answer in task
    public function deleteAnswerInTask($id){
        $record = QuestionAnswer::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // get data sheet listing api
    public function activeDataSheetListingApi(){
        $branch_id = Auth::user()->branch_id;
        if(Auth::user()->role=="Executive"){
            $branch_id = Auth::user()->id;
        }

//      $records = DataSheet::where("status", 1)->orderBy('created_at','DESC')->get();
        $assignedCategories = UserCategory::where('user_id', $branch_id)->pluck("category_id");
//      $records = DataSheet::where("status", 1)->whereIn('id', $assignedSheets->unique())->orderBy('created_at','DESC')->get();
        $records = DataSheet::where("status", 1)->orderBy('created_at','DESC');
        if (count($assignedCategories)) {
            $assignedSheets = DataSheetCategory::whereIn('category_id', $assignedCategories)->pluck("data_sheet_id");
            $records = $records->whereIn('id', $assignedSheets->unique());
        }
        $records = $records->get();

        return HelperServices::returnJsonResponse(1, "Success", $records);
    }

    // get data sheet tasks api
    public function dataSheetTasks($id){
        $record = DataSheet::select("id", "name", "add_additional_msg")->where(["status" => 1, "id" => $id])->first();
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Data sheet not found.");
        }
        $record = $record->load(['tasks' => function ($query) {
                        $query->select('id', 'title', 'data_sheet_id')->where('status', 1);
                    }, "tasks.answers" => function ($query){
                        $query->select('sheet_answers.id', 'sheet_answers.answer');
                    }]);
        return HelperServices::returnJsonResponse(1, "Success", $record);
    }

    // generate progress sheet note api
    public function generateSheetProgressNote($request){
        $customMsgs = [
            'sheet_id.required' => 'Please select data sheet.',
            'client_id.required' => 'Please select client.',
            'sheet_data.required' => 'Error: task data not found.',
            'sheet_data.*.task_id.required' => 'Error: task id not selected.',
            'sheet_data.*.answer_id.required' => 'Please select answer.',
        ];

        $validator = Validator::make(json_decode(json_encode($request),1),[
            'sheet_id' => 'required',
            'client_id' => 'required',
            'sheet_data' => 'required',
            'sheet_data.*.task_id'=>'required',
            'sheet_data.*.answer_id'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 200);
        }

        // get data sheet record
        $record = DataSheet::where(["status" => 1, "id" => $request->sheet_id])->first();
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Data sheet not found.");
        }

        // get client data
        $client = User::where(["id" => $request->client_id, "status" => 1, "role" => "Client"])->first();
        if(!$client){
            return HelperServices::returnJsonResponse(0, "Client not found.");
        }
        $clientFullName = $client->name;
        list($clientFirstName) = explode(' ', $clientFullName);
        $clientGenderNoun = ($client->gender == "Male")? "He": "She";

        $clientHisHerGenderNoun = ($client->gender == "Male")? "His": "Her";
        $clientHimHerObjective = $client->gender == 'Male' ? 'Him' : 'Her';

        // get question id from request
        $qtnArr = array_column($request->sheet_data, 'task_id');
        // get answer id from request
        $ansArr = array_column($request->sheet_data, 'answer_id');
        // get question data from db
        $questions_results = SheetQuestion::select('progress_note', 'id','progress_note_independent', 'assessment', 'plan', 'independent_assessment', 'independent_plan', 'is_new')
            ->whereIn("id", $qtnArr)->get()->toArray();

        /*Added by jehanzeb start*/
        $questions = [];
        foreach ($questions_results as $questions_row){
            if ($questions_row['is_new']) {
                $progressNotesArr = json_decode($questions_row['progress_note']);
                $randomNote = $progressNotesArr[array_rand($progressNotesArr)];
                $progressNotesIndArr = json_decode($questions_row['progress_note_independent']);
                $randomNoteInd = $progressNotesIndArr[array_rand($progressNotesIndArr)];
            } else {
                $randomNote = $questions_row['progress_note'];
                $randomNoteInd = $questions_row['progress_note_independent'];
            }
            $questions[$questions_row['id']]['note'] = $randomNote;
            $questions[$questions_row['id']]['assessment'] = $questions_row['assessment'];
            $questions[$questions_row['id']]['plan'] = $questions_row['plan'];

            $questions_ind[$questions_row['id']]['note'] = $randomNoteInd;
            $questions_ind[$questions_row['id']]['assessment'] = $questions_row['independent_assessment'];
            $questions_ind[$questions_row['id']]['plan'] = $questions_row['independent_plan'];
        }
        /*Added by jehanzeb end*/
        // get answer data from db
        $answers = SheetAnswer::whereIn("id", $ansArr)->pluck("answer", "id")->toArray();

        $progressNote = "<p>";
        $assessmentText = $planText = '';
        $index = $aIndex = $pIndex = 0;
        $alternativeAnswerIdsArr = ['1', '7', '9', '11'];

        $replacements = [
            "##ClientFullName##" => $clientFullName, "##ClientFirstName##" => $clientFirstName,
            "##ClientGenderNoun##" => $clientGenderNoun, "##HIS-HER-GenderNoun##" => $clientHisHerGenderNoun,
            "##HIM-HER-Objective##" => $clientHimHerObjective
        ];

        foreach($request->sheet_data as $data){
            $data = (array)$data;
            $index++;

//          $answerPrompt = (isset($answers[$data["answer_id"]]))? strtolower($answers[$data["answer_id"]])." prompt ": "";
            $answerPrompt = (isset($answers[$data["answer_id"]])) ? strtolower($answers[$data["answer_id"]]) : '';
            if ($answerPrompt && (int)$data["answer_id"] <= 5) {
                $answerPrompt .= ' prompt ';
            }

            $note = $assessment = $plan = '';
            if (in_array($data["answer_id"], $alternativeAnswerIdsArr)) {
                if (isset($questions_ind[$data["task_id"]])) {
                    $note = $questions_ind[$data["task_id"]]['note'];
                    $assessment = $questions_ind[$data["task_id"]]['assessment'];
                    $plan = $questions_ind[$data["task_id"]]['plan'];
                }
//                $note = (isset($questions_ind[$data["task_id"]]))? $questions_ind[$data["task_id"]]['note']: "";
            }else{
                if (isset($questions[$data["task_id"]])) {
                    $note = $questions[$data["task_id"]]['note'];
                    $assessment = $questions[$data["task_id"]]['assessment'];
                    $plan = $questions[$data["task_id"]]['plan'];
                }
//                $note = (isset($questions[$data["task_id"]]))? $questions[$data["task_id"]]['note']: "";
            }

//            $note = str_replace("##ClientFullName##", $clientFullName, $note);  // replace client full name
//            $note = str_replace("##ClientFirstName##", $clientFirstName, $note);  // replace client first name
//            $note = str_replace("##ClientGenderNoun##", $clientGenderNoun, $note);  // replace client gender noun
//            $note = str_replace("##HIS-HER-GenderNoun##", $clientHisHerGenderNoun, $note);  // replace client HIS / HER gender noun
//            $note = str_replace("##HIM-HER-Objective##", $clientHimHerObjective, $note);  // replace client HIM / HER gender objective
//            $note = str_replace("##AnswerPrompt##", $answerPrompt, $note);  // replace selected answer prompt
            $replacements["##AnswerPrompt##"] = $answerPrompt;
            $note = ucfirst(strtolower(str_replace(array_keys($replacements), array_values($replacements), $note)));

            //capitalize every letter after a . ? and ! followed by space or no space
//            $note = ucfirst(strtolower($note));
            $note = preg_replace_callback('/[.!?]\s*.*?\w/',
                function ($matches) { return strtoupper($matches[0]); }, $note);
            $note = str_replace([strtolower($clientFullName), strtolower($clientFirstName), ' i '], [ucwords($clientFullName), ucwords($clientFirstName), ' I '], $note);
            $progressNote .= $note." ";
            if($index % 3 == 0){
                $progressNote .= "</p><p>";
            }

            if ($assessment) {
                $aIndex++;
                $assessment = ucfirst(strtolower(str_replace(array_keys($replacements), array_values($replacements), $assessment)));
                //capitalize every letter after a . ? and ! followed by space or no space
                $assessment = preg_replace_callback('/[.!?]\s*.*?\w/',
                    function ($match) { return strtoupper($match[0]); }, $assessment);
                $assessment = str_replace([strtolower($clientFullName), strtolower($clientFirstName), ' i '], [ucwords($clientFullName), ucwords($clientFirstName), ' I '], $assessment);
                $assessmentText .= $assessment . ' ';
                if ($aIndex % 3 == 0) {
                    $assessmentText .= "</p><p>";
                }
            }

            if ($plan) {
                $pIndex++;
                $plan = ucfirst(strtolower(str_replace(array_keys($replacements), array_values($replacements), $plan)));
                //capitalize every letter after a . ? and ! followed by space or no space
                $plan = preg_replace_callback('/[.!?]\s*.*?\w/',
                    function ($match) { return strtoupper($match[0]); }, $plan);
                $plan = str_replace([strtolower($clientFullName), strtolower($clientFirstName), ' i '], [ucwords($clientFullName), ucwords($clientFirstName), ' I '], $plan);
                $planText .= $plan . ' ';
                if ($pIndex % 3 == 0) {
                    $planText .= "</p><p>";
                }
            }

        }
        /*$sender = Auth::user()->name;
        $executive =User::find(Auth::user()->branch_id);
        $data = array('name'=>$executive->name, "submittedBy"=>$sender,"progressNote"=>$progressNote,"email"=>'muhammadsajidjaved@gmail.com');

        Mail::send(['html'=>'mail'], $data, function($message) use($data) {
            $message->to($data['email'],$data['name'])->subject('Task Submitted By '.$data['submittedBy']);
            $message->from('info@bbcareadmin.com','BBCareAdmin');
        });*/

        $progressNote .= "</p>";
        if (trim($request->add_additional_msg) && trim($request->add_additional_msg) != 'Enter additional details here')
            $progressNote .= "<p>" . trim($request->add_additional_msg) . "</p>";

        if ($assessmentText) {
            $assessmentText = '<br><p class="text-black font-w600">Assessment:</p>' . '<p>' . $assessmentText . '</p>';
            $progressNote .= $assessmentText;
        }

        if ($planText) {
            $planText = '<br><p class="text-black font-w600">Plan:</p>' . '<p>' . $planText . '</p>';
            $progressNote .= $planText;
        }

        return HelperServices::returnJsonResponse(1, "Success", ["progress_note" => $progressNote]);
    }

    // save data sheet tasks & note api
    public function saveSheetTasks($request){
        $customMsgs = [
            'sheet_id.required' => 'Please select data sheet.',
            'client_id.required' => 'Please select client.',
            'sheet_data.required' => 'Error: task data not found.',
            'sheet_data.*.task_id.required' => 'Error: task id not selected.',
            'sheet_data.*.answer_id.required' => 'Please select answer.',
            'progress_note.required' => 'Error: progress note not found.',
        ];
        $request = json_decode(json_encode($request),1);
        $validator = Validator::make((array)$request,[
            'sheet_id' => 'required',
            'client_id' => 'required',
            'sheet_data' => 'required',
            'sheet_data.*.task_id'=>'required',
            'sheet_data.*.answer_id'=>'required',
            'progress_note' => 'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 200);
        }

        // get data sheet record
        $record = DataSheet::where(["status" => 1, "id" => $request['sheet_id']])->first();
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Data sheet not found.");
        }

        // get client data
        $client = User::where(["id" => $request['client_id'], "status" => 1, "role" => "Client"])->first();

        if(!$client){
            return HelperServices::returnJsonResponse(0, "Client not found.");
        }
        $request['add_additional_msg'] = trim($request['add_additional_msg']) && trim($request['add_additional_msg']) != 'Enter additional details here' ? trim($request['add_additional_msg']) : '';
        // save records in db
        $dbStatus = $this->saveTaskDataInDB($record->id, $client->id, $request['progress_note'], $request['sheet_data'], $request['add_additional_msg']);
        if($dbStatus){
            $sheet_name = $record->name;
            $sender = Auth::user()->name;
            $executive = User::find($client->branch_id);
            $coordinator = User::find($executive->branch_id);
            $data = ['name' => $executive->name, "email" => $executive->email, 'coordinatorName' => $coordinator->name, 'coordinatorEmail' => $coordinator->email, 'submittedBy' => $sender, 'clientName' => $client->name, 'SheetName' => $sheet_name, 'progressNote' => $request['progress_note']];
            sendEmail($data);
//            Mail::send(['html'=>'mail'], $data, function($message) use($data) {
//                $message->to($data['email'], $data['name'])->cc([$data['coordinatorEmail']])->subject('Task Submitted By ' . $data['submittedBy']);
//                $message->from('info@bbcareadmin.com', 'BBCareAdmin');
//            });

            return HelperServices::returnJsonResponse(1, "Tasks added successfully");
        } else {
            return HelperServices::returnJsonResponse(0, "Something went wrong");
        }
    }

    // save task report data in database
    private function saveTaskDataInDB($sheetId, $clientId, $progressNote, $taskData = array(), $additionalMsg = ""){
        $currentUser = request()->user();
        $record = new TaskReport();
        $record->data_sheet_id = $sheetId;
        $record->staff_id = $currentUser->id;
        $record->client_id = $clientId;
        $record->additional_message = $additionalMsg;
        $record->progress_note = $progressNote;
        if($record->save()){

            $dataArr = [];
            foreach ($taskData as $data){
                $dataArr[] = new ReportData(["sheet_question_id" => $data["task_id"], "sheet_answer_id" => $data["answer_id"]]);
            }
            if(sizeof($dataArr) > 0) {
                $record->reportData()->saveMany($dataArr);
            }


            return true;
        } else {
            return false;
        }
    }

//    Data sheet categories
    public function assignCategoryToSheetRequest($request){
        $customMsgs = [
            'id.required' => 'Something went wrong.',
            'categories.required' => 'Please select at-least one category.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'categories'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = DataSheet::where(["id" => $request->id])->first();
        if ($record) {
            $record->categories()->attach($request->categories);
            // assign category to user
            return HelperServices::returnJsonResponse(1, 'Categories assigned successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Invalid request.');
        }
    }

    // remove category from user
    public function removeCategoryFromSheet($id){
        $record = DataSheetCategory::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Invalid request.');
        }
        $record->delete();
        return HelperServices::returnJsonResponse(1, 'Category successfully removed.');
    }
}
