<?php


namespace App\Services;

use App\Permission;
use App\StaffPivotClient;
use App\User;
use http\Env\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;

class UserService{

    // Datatable filter users
    public function userFilterListAjax($type, $branch='')
    {
        $data = User::query();
        if ($type != "all") {
            $data = $data->where("role", ucfirst($type));
        }

        if ($branch) {
            $data = $data->where("branch_id", $branch);
        }

        return $this->returnDatatableResponse($data);
    }
    public function userBranchListAjax($type,$branch){
        $data = User::query();
        if($type != "all"){
            $data = $data->where("role", ucfirst($type));
        }
        if($branch) {
            $data = $data->where("branch_id", $branch);
        }



        return $this->returnDatatableResponse($data,$type);
    }

    private function returnDatatableResponse($data){
        return datatables()->of($data)
            ->addColumn('action', function($row){
                $btn = '<div class="d-flex">';
                $user_type = strtolower($row->role);
                if(Auth::user()->role == 'SuperAdmin' and $row->role=='Executive') {
                    $btn .= '<a href="' . route("portal.userFilterList", ["type" => 'admin',"id" => $row->id]) . '" class="btn btn-info info shadow sharp mr-1"><i class="fa fa-user"></i> Admin</a>';
                    $btn .= '<a href="' . route("portal.userFilterList", ["type" => 'staff',"id" => $row->id]) . '" class="btn btn-warning warning shadow sharp mr-1"><i class="fa fa-users"></i> Staff</a>';
                    $btn .= '<a href="' . route("portal.userFilterList", ["type" => 'client',"id" => $row->id]) . '" class="btn btn-default shadow sharp mr-1"><i class="fa fa-user-circle"></i> Client</a>';
                }
                if(checkPermission('viewUser', $user_type)) {
                    $btn .= '<a href="' . route("portal.viewUserPage", ["id" => $row->id]) . '" class="btn btn-success shadow sharp mr-1"><i class="fa fa-cog"></i> Operations</a>';
                }
                if(checkPermission('editUser',$user_type)) {
                    $btn .= '<a href="' . route("portal.editUserPage", ["id" => $row->id]) . '" class="btn btn-primary shadow sharp mr-1"><i class="fa fa-pencil"></i> Edit</a>';
                }
                if ($row->id != Auth::id() and checkPermission('deleteUser',$user_type)) {
                        $btn .= '<a href="' . route("portal.deleteUser", ["id" => $row->id]) . '" class="deletedBtn btn btn-danger shadow sharp"><i class="fa fa-trash"></i> Delete</a>';
                }

                $btn .= '</div>';
                return $btn;
            })
            ->editColumn('created_at', function ($row){
                $user_type = strtolower($row->role);
                return $row->created_at ;
            })
            ->rawColumns(['action'])
            ->addIndexColumn()
            ->make(true);
    }

    // add new user
    public function addNewUserRequest($request){
        //print_r($request->all());exit;
        $customMsgs = [
            'name.required' => 'Please enter name.',
            'email.required' => 'Please enter email.',
            'email.email' => 'Please enter valid email address.',
            'email.unique' => 'User already exist with this email.',
            'username.required' => 'Please enter email.',
            'username.unique' => 'User already exist with this username.',
            'password.required' => 'Please enter password.',
            'role.required' => 'Please select role.',
            'employee_id.required_if' => 'Please enter employee id.',
            'ddd_id.required_if' => 'Please enter ddd id.',
            'thumbnail.mimetypes' => 'Only JPG or PNG file is allowed.',
            'thumbnail.max' => 'Maximum uploading profile image size limit is 1MB.',
        ];
        $rules = [
            'name'=>'required',
            'email'=>'required|email|unique:users',
            'username'=>'required|unique:users',
            'password'=>'required',
            'role'=>'required',
            'employee_id'=>'required_if:role,==,Staff',
            'ddd_id'=>'required_if:role,==,Client',
            'thumbnail'=>'nullable|mimetypes:image/jpeg,image/png|max:1000',
        ];
        $data = [];
        if($request->role == 'Client'){
            $rules['username'] = '';
            $rules['password'] = '';
            $data['username'] = 'client-'.rand(1000,9999).'-'.time();
            $data['password'] = 'nopass';
            $request->password = $data['password'];
        }



        $validator = Validator::make($request->all(),$rules, $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }


        $data['password'] = bcrypt($request->password);
        if($request->file('thumbnail')){
            $profileFile = $request->file('thumbnail');
            $profileFileExtension = $profileFile->getClientOriginalExtension();
            $profileImgName = 'profile-img'.time().'.'.$profileFileExtension;

            $data['thumbnail'] = $profileImgName;
        }
        $record = User::create($request->except("password", "thumbnail") + $data);
        if($record){
            $editUrl = route("portal.editUserPage", ["id" => $record->id]);

            if($request->role=='Executive'){
                $dataU['branch_id'] =$record->id;
            }else if(Auth::user()->role == 'Executive'){
                $dataU['branch_id'] =Auth::user()->id;
            }else{
                $dataU['branch_id'] =$request->branch_id;
            }
            User::Where('id', $record->id)->update($dataU);
            if(isset($profileFile)){
                $uploadingPath = public_path('/storage/user-data/user-'.$record->id);
                $imageUpload = $profileFile->move($uploadingPath, $profileImgName);
                if($imageUpload){
                    return HelperServices::returnJsonResponse(1, 'User created successfully.', ["edit_url" => $editUrl]);
                } else {
                    User::where(['id' => $record->id])->update(["thumbnail" => ""]);
                    return HelperServices::returnJsonResponse(1, 'Sorry, there was an error uploading your image but user created successfully', ["edit_url" => $editUrl]);
                }
            }
            return HelperServices::returnJsonResponse(1, 'User created successfully.', ["edit_url" => $editUrl]);
        } else {
            return HelperServices::returnJsonResponse(0, 'User not created.');
        }
    }

    public function addUserPermissions($request){

        $data['role']=$request->role;
        Permission::where('role',$request->role)->delete(); // Delete all role permission
        //dd($request->user_permissions);
        foreach($request->user_permissions as $key => $value){
           // Permission::create();
            $permission = new Permission;
            $permission->name = $value;
            $permission->role_key=$value;
            $permission->role=$request->role;
            $permission->save();
        }
        return HelperServices::returnJsonResponse(1, 'Role permission updated successfully.');

    }

    // edit user page
    public function editUserPage($id){
        $user = User::find($id);
        $branchName='';
        if($user->branch_id){
            $branchName = User::find($user->branch_id)->name;
        }
        if(!$user){
            abort(404);
        }
        return view("portal.users.editUser", compact("user","branchName"));
    }

    // update user request
    public function updateUserRequest($request){
        $customMsgs = [
            'id.required' => 'Something went wrong.',
            'name.required' => 'Please enter name.',
            'email.required' => 'Please enter email.',
            'email.email' => 'Please enter valid email address.',
            'email.unique' => 'User already exist with this email.',
            'username.required' => 'Please enter username.',
            'username.unique' => 'User already exist with this username.',
            'thumbnail.mimetypes' => 'Only JPG or PNG file is allowed.',
            'thumbnail.max' => 'Maximum uploading profile image size limit is 1MB.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'name'=>'required',
            'email'=>'required|email|unique:users,email,'.$request->id,
            'username'=>'required|unique:users,username,'.$request->id,
            'thumbnail'=>'nullable|mimetypes:image/jpeg,image/png|max:1000',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $data = array();
        if($request->password != ""){
            $data['password'] = bcrypt($request->password);
        }
        if($request->file('thumbnail')){
            $profileFile = $request->file('thumbnail');
            $profileFileExtension = $profileFile->getClientOriginalExtension();
            $profileImgName = 'profile-img'.time().'.'.$profileFileExtension;

            $data['thumbnail'] = $profileImgName;
        }
        $data['updated_at'] = new \DateTime();
        if(User::where(['id' => $request->id])->update($request->except("id", "password", "thumbnail") + $data)){

            $exe_user = User::find($request->id);

            if($exe_user->role=='Executive'){
                $data['status']=$request->status;
                User::Where('branch_id', $exe_user->id)->update($data);
            }



            if(isset($profileFile)){
                $uploadingPath = public_path('/storage/user-data/user-'.$request->id);
                $imageUpload = $profileFile->move($uploadingPath, $profileImgName);
                if($imageUpload){
                    return HelperServices::returnJsonResponse(1, 'User updated successfully');
                } else {
                    User::where(['id' => $request->id])->update(["thumbnail" => ""]);
                    return HelperServices::returnJsonResponse(1, 'Sorry, there was an error uploading your image but user updated successfully');
                }
            }
            return HelperServices::returnJsonResponse(1, 'User updated successfully');
        } else {
            return HelperServices::returnJsonResponse(0, 'Nothing change');
        }
    }

    // view user page
    public function viewUserPage($id){
        $user = User::find($id);
        $branchName='';
        if($user->branch_id){
            $branchName = User::find($user->branch_id)->name;
        }
        if(!$user){
            abort(404);
        }
        $clients = [];
        if($user->role == "Staff"){
            $alreadyAssClients = StaffPivotClient::where('staff_id','=',$id)->pluck("client_id");
            //echo '<pre>';
            //print_r($alreadyAssClients);exit;

            $clients = User::select("id", "name", "username")->where("role", "Client")->whereNotIn("id", $alreadyAssClients)->get();
        }
        return view("portal.users.viewUser", compact("user", "clients","branchName"));
    }

    // delete user
    public function deleteUser($id){
        $user = User::find($id);
        if(!$user){
            return HelperServices::returnJsonResponse(0, 'User not found.');
        }
        if($user->delete()){
            return HelperServices::returnJsonResponse(1, 'User deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'User not deleted.');
        }
    }

    public function assignClientToStaffRequest($request){
        $customMsgs = [
            'id.required' => 'Something went wrong.',
            'clients.required' => 'Please select client.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'clients'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = User::where(["id" => $request->id, "role" => "Staff"])->first();
        if($record){
            $record->staffClients()->attach($request->clients);
    // assign client to staff member
            return HelperServices::returnJsonResponse(1, 'Clients assigned successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Clients only assign to staff member.');
        }
    }

    // remove client from staff
    public function removeClientFromStaff($id){
        $record = StaffPivotClient::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Relationship not found.');
        }
        $record->delete();
        return HelperServices::returnJsonResponse(1, 'Client successfully removed.');
    }

    // get staff profile detail
    public function getStaffProfile($user, $returnJson = true){
        if(is_object($user)){
            $userDetail = $user;
        } else {
            $userDetail = User::find($user);
        }
        if($returnJson){
            if(!$userDetail){
                return HelperServices::returnJsonResponse(0, 'Record not found');
            }
            $userDetail = HelperServices::arrayLoopGenModified($userDetail->toArray());
            return HelperServices::returnJsonResponse(1, 'Success', $userDetail);
        } else {
            return $userDetail;
        }
    }

    // update profile detail
    public function staffProfileUpdate($request){
        $currentUser = request()->user();
        $customMsgs = [
            'name.required' => 'Please enter name.',
            'email.required' => 'Please enter email.',
            'email.email' => 'Please enter valid email address.',
            'email.unique' => 'User already exist with this email.'
        ];
        $validator = Validator::make($request->all(),[
            'name'=>'required',
            'email'=>'required|email|unique:users,email,'.$currentUser->id,
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray);
        }

        $data = array();
        if($request->password != ""){
            $data['password'] = bcrypt($request->password);
        }
        $data['updated_at'] = new \DateTime();
        if(User::where(['id' => $currentUser->id])->update($request->only("name", "email", "phone_number", "gender", "address", "emergency_contact") + $data)){
            $profileDetail = $this->getStaffProfile($currentUser->id, false);
            $profileDetail = HelperServices::arrayLoopGenModified($profileDetail->toArray());
            return HelperServices::returnJsonResponse(1, 'Profile updated successfully', $profileDetail);
        } else {
            return HelperServices::returnJsonResponse(0, 'Nothing change.');
        }
    }

    // profile image update
    public function updateProfileImg($request){
        $currentUser = request()->user();
        $customMsgs = [
            'thumbnail.required' => 'Please select image.',
            'thumbnail.mimetypes' => 'Only JPG or PNG file is allowed.',
            'thumbnail.max' => 'Maximum uploading profile image size limit is 1MB.',
        ];
        $validator = Validator::make($request->all(),[
            'thumbnail'=>'required|mimetypes:image/jpeg,image/png|max:1000',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray);
        }

        $profileFile = $request->file('thumbnail');
        $profileFileExtension = $profileFile->getClientOriginalExtension();
        $profileImgName = 'profile-img'.time().'.'.$profileFileExtension;
        $uploadingPath = public_path('/storage/user-data/user-'.$currentUser->id);
        $imageUpload = $profileFile->move($uploadingPath, $profileImgName);
        if($imageUpload){
            $data = [];
            $data['thumbnail'] = $profileImgName;
            $data['updated_at'] = new \DateTime();
            User::where(['id' => $currentUser->id])->update($data);
            //$proImg = asset('storage/user-data/user-'.$currentUser->id.'/'.$profileImgName);
            $profileDetail = $this->getStaffProfile($currentUser->id, false);
            $profileDetail = HelperServices::arrayLoopGenModified($profileDetail->toArray());
            return HelperServices::returnJsonResponse(1, 'Profile image updated successfully', $profileDetail);
        } else {
            return HelperServices::returnJsonResponse(0, 'Nothing change.');
        }
    }

    // get client listing of staff member
    public function staffClientListing($request, $userId, $type){

        $page = ($request->get('page'))? $request->get('page') : '1';
        $type = $request->get('type');
        if($type=='Staff'){
            $staff = User::find($userId);
            $clients = $staff->staffClients()->where('status','=','1')->paginate(env('API_PER_PAGE_LIMIT'))->toArray();
        }elseif($type=='Executive' || $type=='Admin') {
            $user = User::find($userId);
            $branch_id = $user->branch_id;
            if ($type == 'Executive') {
                $branch_id = $user->id;
            }
            $clientsD = User::where('branch_id', $branch_id)->where('role', 'Client')->where('status', 1)->paginate(env('API_PER_PAGE_LIMIT'));
            $clientsDD = [];
            foreach ($clientsD as $row) {
                $row['pivot'] = DB::table('staff_pivot_clients')->where('client_id', $row->id)->first();
                $clientsDD[] = $row;
            }
            $clientsD = $clientsD->toArray();

            $clients['current_page']= $clientsD['current_page'];
            $clients['data']= $clientsDD;
            $clients['first_page_url']= $clientsD['first_page_url'];
            $clients['from']= $clientsD['from'];
            $clients['last_page']= $clientsD['last_page'];
            $clients['last_page_url']= $clientsD['last_page_url'];
            $clients['next_page_url']= $clientsD['next_page_url'];
            $clients['path']= $clientsD['path'];
            $clients['per_page']= $clientsD['per_page'];
            $clients['prev_page_url']= $clientsD['prev_page_url'];
            $clients['total']= $clientsD['total'];
            $clients['to']= $clientsD['to'];
        }

        $clients = HelperServices::arrayLoopGenModified($clients);
        return HelperServices::returnJsonResponse(1, 'Success', $clients);
    }
}
