<?php


namespace App\Services;


use App\Scopes\LatestScope;
use App\TaskReport;
use App\User;
use Carbon\Carbon;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Arr;
use PDF;

class TaskReportService
{
    // Datatable filter task history of user
    public function userTaskHistoryAjaxList($userId){
        $data = TaskReport::with('client', 'sheet')->where(["staff_id" => $userId])->withoutGlobalScope(new LatestScope());

        return $this->returnDatatableResponse($data);
    }

    private function returnDatatableResponse($data){
        return datatables()->of($data)
            ->addColumn('action', function($row){
                $btn = '<div class="d-flex">';
                $btn .=  '<a href="'.route("portal.viewTaskReportPage", ["id" => $row->id]).'" class="btn btn-success shadow sharp mr-1"><i class="fa fa-eye"></i> View</a>';
                $btn .= '</div>';
                return $btn;
            })
            ->editColumn('created_at', function ($row){
                return Carbon::parse($row->created_at)->format("d M. Y h:i a");
            })
            ->rawColumns(['action'])
            ->addIndexColumn()
            ->make(true);
    }

    // get task report detail
    public function viewTaskReportPage($reportId)
    {
        $data = TaskReport::find($reportId);
        if (!$data) {
            abort(404);
        }
        $data = $data->load('client', 'sheet', 'staff', 'reportData.question', 'reportData.answer');

        return view('portal.taskReports.viewTaskReport', compact("data"));
    }

    // get client listing of staff member api
    public function getSheetReports($request, $user)
    {
        $userId = $user->id;
        $page = ($request->get('page')) ? $request->get('page') : '1';
        $whereConditionArr = $user->role == 'Client' ? ["client_id" => $userId] : ["staff_id" => $userId];
        $taskReports = TaskReport::with('client', 'sheet', 'staff')->where($whereConditionArr)->paginate(env('API_PER_PAGE_LIMIT'));
        $reportData = [];
        foreach ($taskReports as $report) {
            $reportData[] = $this->sheetReportObj($report);
        }

        $returnData = HelperServices::returnPaginationArr($taskReports, $reportData);
        $returnData = HelperServices::arrayLoopGenModified($returnData);
        return HelperServices::returnJsonResponse(1, 'Success', $returnData);
    }

    // sheet report obj
    private function sheetReportObj($record)
    {
        $returnObj = [
            "id" => $record->id,    // report id
            "sheet_id" => $record->data_sheet_id,
            "sheet_name" => $record->sheet->name,
            "client_id" => $record->client_id,
            "client_name" => $record->client->name,
            "staff_id" => $record->staff_id,
            "staff_name" => $record->staff->name,
            "created_at" => $record->created_at->format("Y-m-d H:i:s"),
        ];
        return $returnObj;
    }

    public function taskHistoryReportPage() {
        $user = Auth::user();
        $users = User::select('id', 'name', 'email', 'role')->where('status', 1);
        $clients = [];
        if ($user->role == 'SuperAdmin') {
            $users = $users->where('role', 'Coordinator')->get();
        } else if ($user->role == 'Coordinator') {
            $users = $users->where(['branch_id' => $user->id, 'role' => 'Executive'])->get();
        } else if ($user->role == 'Executive') {
            $users = $users->where('branch_id', $user->id)->whereIn('role', ['Manager', 'Admin', 'Staff'])->get();
            $clients = User::select('id', 'name', 'email', 'role')->where(['branch_id' => $user->id, 'role' => 'Client', 'status' => 1])->get();
        } else {
            $authMessage = "Not Authorized! Please contact with the SuperAdmin";
            return view("auth.login", compact('authMessage'));
        }
        return view("portal.taskReports.history-report", compact('users', 'clients'));
    }

    public function taskHistoryReportRequest($request)
    {
        $startDate = Carbon::parse($request->start)->format('Y-m-d 00:00:00');
        $endDate = Carbon::parse($request->end)->format('Y-m-d 23:59:59');
        $reportType = $request->reportType;
        $reports = TaskReport::with('client', 'sheet')->whereBetween('created_at', [$startDate, $endDate])->withoutGlobalScope(new LatestScope());
        $whereInArr = [];
        if ($reportType == '1') {
            $isWhereInApplied = false;
            if ($request->staff) {
                $whereInArr = explode(',', $request->staff);
//                $usersArray = explode(',', $request->staff);
//                $whereInArr = Arr::collapse([[$request->executive], $usersArray]);
                $isWhereInApplied = true;
            } else if ($request->executive) {
                $executiveUsersArr = User::where(['branch_id' => $request->executive, 'status' => 1])->whereIn('role', ['Manager', 'Admin', 'Staff'])->pluck("id");
                $whereInArr = Arr::collapse([[$request->executive], $executiveUsersArr]);
                $isWhereInApplied = true;
            } else if ($request->coordinator) {
                $executivesArr = User::where(['branch_id' => $request->coordinator, 'role' => 'Executive', 'status' => 1])->pluck('id');
                $executiveUsersArr = User::where('status', 1)->whereIn('branch_id', $executivesArr)->whereIn('role', ['Manager', 'Admin', 'Staff'])->pluck('id');
                $whereInArr = Arr::collapse([$executivesArr, $executiveUsersArr]);
                $isWhereInApplied = true;
            }
            if ($isWhereInApplied) {
                $reports = $reports->whereIn('staff_id', $whereInArr);
            }
        } else {
            $isWhereInApplied = false;
            if ($request->client) {
                $whereInArr = explode(',', $request->client);
                $isWhereInApplied = true;
            } else if ($request->executive) {
                $whereInArr = User::where(['branch_id' => $request->executive, 'role' => 'Client', 'status' => 1])->pluck('id');
                $isWhereInApplied = true;
            } else if ($request->coordinator) {
                $executivesArr = User::where(['branch_id' => $request->coordinator, 'role' => 'Executive', 'status' => 1])->pluck('id');
                $whereInArr = User::where(['role' => 'Client', 'status' => 1])->whereIn('branch_id', $executivesArr)->pluck('id');
                $isWhereInApplied = true;
            }
            if ($isWhereInApplied) {
                $reports = $reports->whereIn('client_id', $whereInArr);
            }
        }
        $reports = $reports->get();
        $reportsArr = $reports->toArray();
        $staffIdsArr = array_unique(array_column($reportsArr, 'staff_id'));
        $clientIdsArr = array_unique(array_column($reportsArr, 'client_id'));
        $stats = ['reportCount' => count($reports), 'staffCount' => count($staffIdsArr), 'clientCount' => count($clientIdsArr)];
        $dateRange = $request->start . ' - ' . $request->end;
        $data = view('portal.taskReports.history-report-data', compact('reports', 'reportType', 'stats', 'dateRange'))->render();
        echo $data; exit();
    }

    public function getTaskHistoryReportCount($startDate, $endDate, $whereCondition = [])
    {
        return TaskReport::where($whereCondition)->whereBetween('created_at', [$startDate, $endDate])->withoutGlobalScope(new LatestScope())->count();
    }

    public function clientTaskHistoryPDFRequest($request) {
        set_time_limit(0);
        $startDate = Carbon::parse($request->start)->format('Y-m-d 00:00:00');
        $endDate = Carbon::parse($request->end)->format('Y-m-d 23:59:59');
        $user = User::find($request->client);
        $reports = TaskReport::with('sheet')->where(['client_id' => $request->client])->whereBetween('created_at', [$startDate, $endDate])->withoutGlobalScope(new LatestScope())->get();
        $dateRange = Carbon::parse($request->start)->format('m/d/Y') . ' - ' . Carbon::parse($request->end)->format('m/d/Y');
        $pdf = PDF::loadView('portal.taskReports.clientTaskHistory', compact('reports', 'user', 'dateRange'));
        return $pdf->stream('bb-care-notes.pdf');
    }
}
