<?php


namespace App\Services;


use App\DataSheet;
use App\QuestionAnswer;
use App\SheetAnswer;
use App\SheetQuestion;
use Illuminate\Support\Facades\Validator;

class DataSheetService
{
    // data sheet listing page
    public function dataSheetPage(){
        $records = DataSheet::withCount("tasks")->get();
        return view("portal.dataSheets.index", compact("records"));
    }

    // add new data sheet request
    public function addNewDataSheetRequest($request){
        $customMsgs = [
            'name.required' => 'Please enter sheet name.',
        ];
        $validator = Validator::make($request->all(),[
            'name'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = new DataSheet();
        $record->name = $request->name;
        $record->add_additional_msg = $request->add_additional_msg;
        if($record->save()){
            $redirectUrl = route("portal.editDataSheetPage", ["id" => $record->id]);
            return HelperServices::returnJsonResponse(1, 'Data sheet created successfully.', ["redirect_url" => $redirectUrl]);
        } else {
            return HelperServices::returnJsonResponse(0, 'Data sheet not created.');
        }
    }

    // get data sheet page
    public function editDataSheet($id){
        $record = DataSheet::find($id);
        if(!$record){
            abort(404);
        }
        return view('portal.dataSheets.editDataSheet', compact('record'));
    }

    // update data sheet
    public function updateDataSheetRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'name.required' => 'Please enter sheet name.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'name'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = DataSheet::find($request->id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Record not found");
        }
        $record->name = $request->name;
        $record->add_additional_msg = $request->add_additional_msg;
        $record->status = $request->status;
        if($record->save()){
            return HelperServices::returnJsonResponse(1, "Record updated successfully");
        } else {
            return HelperServices::returnJsonResponse(0, "Nothing updated");
        }
    }

    // view data sheet page
    public function viewDataSheetPage($id){
        $record = DataSheet::find($id);
        if(!$record){
            abort(404);
        }
        $record = $record->load("tasks.answers");
        return view("portal.dataSheets.viewDataSheet", compact("record"));
    }

    // delete data sheet
    public function deleteDataSheet($id){
        $record = DataSheet::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // add sheet task page
    public function addSheetTaskPage($sheetId){
        $record = DataSheet::find($sheetId);
        if(!$record){
            abort(404);
        }
        $sheetId = $record->id;
        return view("portal.dataSheets.sheetTasks.addSheetTask", compact("sheetId"));
    }

    // add sheet task request
    public function addSheetTaskRequest($request){
        $customMsgs = [
            'data_sheet_id.required' => 'Error: Something went wrong.',
            'data_sheet_id.exists' => 'Error: Data Sheet not found.',
            'title.required' => 'Please enter title.',
            'progress_note.required' => 'Please enter progress note.',
        ];
        $validator = Validator::make($request->all(),[
            'data_sheet_id'=>'required|exists:data_sheets,id',
            'title'=>'required',
            'progress_note'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = new SheetQuestion();
        $record->title = $request->title;
        $record->progress_note = $request->progress_note;
        $record->data_sheet_id = $request->data_sheet_id;
        if($record->save()){
            $redirectUrl = route("portal.editSheetTaskPage", ["id" => $record->id]);
            return HelperServices::returnJsonResponse(1, 'Record created successfully.', ["redirect_url" => $redirectUrl]);
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not created.');
        }
    }

    // edit sheet task page
    public function editSheetTaskPage($taskId){
        $record = SheetQuestion::find($taskId);
        if(!$record){
            abort(404);
        }
        return view("portal.dataSheets.sheetTasks.editSheetTask", compact("record"));
    }

    // update sheet task request
    public function updateSheetTaskRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'title.required' => 'Please enter title.',
            'progress_note.required' => 'Please enter progress note.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'title'=>'required',
            'progress_note'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $record = SheetQuestion::find($request->id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        $record->title = $request->title;
        $record->progress_note = $request->progress_note;
        $record->status = $request->status;
        if($record->save()){
            return HelperServices::returnJsonResponse(1, 'Record updated successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not updated.');
        }
    }

    // delete sheet question
    public function deleteSheetTask($id){
        $record = SheetQuestion::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // add answer in task
    public function addAnswerInTaskRequest($request){
        $customMsgs = [
            'id.required' => 'Error: Something went wrong.',
            'answer.required' => 'Please enter answer.',
        ];
        $validator = Validator::make($request->all(),[
            'id'=>'required',
            'answer'=>'required',
        ], $customMsgs);

        if ($validator->fails()) {
            $errorArray = HelperServices::validationErrorFormat($validator->errors());
            return HelperServices::returnJsonResponse(2, 'Validation Errors', [], $errorArray, 422);
        }

        $question = SheetQuestion::find($request->id);
        if(!$question){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }

        $record = new SheetAnswer();
        $record->answer = $request->answer;
        if($record->save()){
            $question->answers()->attach($record->id, ["delete_answer" => 1]);
            return HelperServices::returnJsonResponse(1, 'Record added successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not added.');
        }
    }

    // delete answer in task
    public function deleteAnswerInTask($id){
        $record = QuestionAnswer::find($id);
        if(!$record){
            return HelperServices::returnJsonResponse(0, 'Record not found.');
        }
        if($record->delete()){
            return HelperServices::returnJsonResponse(1, 'Record deleted successfully.');
        } else {
            return HelperServices::returnJsonResponse(0, 'Record not deleted.');
        }
    }

    // get data sheet listing api
    public function activeDataSheetListingApi(){
        $records = DataSheet::where("status", 1)->get();
        return HelperServices::returnJsonResponse(1, "Success", $records);
    }

    // get data sheet tasks api
    public function dataSheetTasks($id){
        $record = DataSheet::select("id", "name", "add_additional_msg")->where(["status" => 1, "id" => $id])->first();
        if(!$record){
            return HelperServices::returnJsonResponse(0, "Data sheet not found.");
        }
        $record = $record->load(['tasks' => function ($query) {
                        $query->select('id', 'title', 'data_sheet_id')->where('status', 1);
                    }, "tasks.answers" => function ($query){
                        $query->select('sheet_answers.id', 'sheet_answers.answer');
                    }]);
        return HelperServices::returnJsonResponse(1, "Success", $record);
    }

}
